/*
** Alias Assembler  Version 1.0  Copyright 1995, 1996 Philippe Verdy
**
** File:       asalias.c
** Purpose:    micro-assembler for alias references into Intel OMF object
** Build:      bcc -O -ms asalias.c
** Usage:      asalias module.als module.obj
** Make rules: ALIASTOOL = $(BCROOT)\bin\asalias
**             .als.obj:
**                 $(ALIASTOOL) $< $@
** Warnings:   usage/error checking is very poor, no filename checking !
**             no default extension, no default module.obj from module.als !
**             module.obj will be overwritten on invocation !
*/
#include <stdio.h>
#include <io.h>
#include <stdlib.h>
#include <string.h>
#include <setjmp.h>

jmp_buf fatalerror;

/***********************************************/
/* structure for OMF (Intel .OBJ/.LIB) records */
/***********************************************/
#pragma option -a1
#pragma pack(1)
struct OMFRecord {
  unsigned char check; /* stored at end of record in OMF file */
  struct {
    unsigned char  type;
    unsigned short len;
  } rec;
  unsigned char buffer[4096];
} OMF; /* global variable, sufficient for only one open OMF record per run */
#pragma option -a.
#pragma pack()

/* open a new OMF record of a given type */
#define OMF_THEADR ((unsigned char)0x80)
#define OMF_LNAMES ((unsigned char)0x96)
#define OMF_COMENT ((unsigned char)0x88)
#define OMF_ALIAS  ((unsigned char)0xC6)
#define OMF_MODEND ((unsigned char)0x8A)
void OMFnew(unsigned char type)
{
  OMF.check = -type;
  OMF.rec.type = type;
  OMF.rec.len = 0;
}

/* add a byte to the current record */
void OMFputc(unsigned char c)
{
  OMF.buffer[OMF.rec.len++] = c;
  OMF.check -= c;
}

/* close the current record */
void OMFstore(FILE *g)
{
  OMFputc(OMF.check);
  if (!fwrite(&OMF.rec, 1, sizeof(OMF.rec) + OMF.rec.len, g))
    longjmp(fatalerror, 3);
}

/* store multiple bytes to the current record */
void OMFputm(unsigned char *m, size_t n)
{
  while (n--) OMFputc(*m++);
}

/* store a Pascal string in the current record */
void OMFputs(char *s)
{
  int l;
  OMFputc((unsigned char)(l = strlen(s)));
  OMFputm((unsigned char*)s, l);
}

/*************************/
/* Atomic OMF operations */
/*************************/

/* Create a THEADR record (object module header) */
void OMFputTHEADR(FILE *g, char *theadr)
{
  OMFnew(OMF_THEADR);
  OMFputs(theadr);
  OMFstore(g);
}

/* add a LNAMES record (link names for segments, groups, externs and so on) */
void OMFputLNAMES(FILE *g, char *names)
{
  OMFnew(OMF_LNAMES);
  while (*names) {
    OMFputs(names);
    names += strlen(names) + 1;
  }
  OMFputs("");
  OMFstore(g);
}

/* add a "comment" record */
/* - comment flags (bitmask) */
#define COMF_PURGE     ((unsigned char)0x40)
/* - comment classes */
#define COMC_OMFEXTEND ((unsigned char)0xA1)
#define COMC_LINKPASS2 ((unsigned char)0xA2)
void OMFputCOMENT(FILE *g, unsigned char comf_flags, unsigned char comc_class,
  unsigned char *extra, size_t extlen)
{
  OMFnew(OMF_COMENT);
  OMFputc(comf_flags);
  OMFputc(comc_class);
  OMFputm(extra, extlen);
  OMFstore(g);
}

/* add an ALIAS record (OMF extension) */
void OMFputALIAS(FILE *g, char *aliasname, char *refname)
{
  OMFnew(OMF_ALIAS);
  OMFputs(aliasname);
  OMFputs(refname);
  OMFstore(g);
}

/* add a MODEND record (object module end) */
void OMFputMODEND(FILE *g, char *modend)
{
  OMFnew(OMF_MODEND);
  OMFputs(modend);
  OMFstore(g);
}

/*********************/
/*********************/

int main(int argc,char *argv[])
{
  FILE         *f, *g;
  char         line[512];
  int          linnum = 0;
  int          errors = 0;
  volatile int fatal = 0;

  fflush(stderr); close(2); dup2(1, 2);
  printf("Alias Assembler  Version 1.0  Copyright 1995, 1996 Philippe Verdy\n");
  if (argc != 3) {
    fprintf(stderr, "Usage: %s module.als module.obj\n",
      argv[0] == NULL ? "asalias" : argv[0]);
    exit(2);
  }
  if ((f=fopen(argv[1], "rt"))==NULL) {
    fprintf(stderr, "**Fatal** "); perror(argv[1]);
    exit(2);
  }
  if ((g=fopen(argv[2], "wb"))==NULL) {
    fclose(f);
    fprintf(stderr, "**Fatal** "); perror(argv[2]);
    exit(2);
  }
  fatal = setjmp(fatalerror);
  if (!fatal) {
    printf("Assembling file: %s to: %s\n", argv[1], argv[2]);
    OMFputTHEADR(g, argv[1]);
    OMFputLNAMES(g, "\0");
    OMFputCOMENT(g, COMF_PURGE, COMC_OMFEXTEND, "", 0);
    while (fgets(line, 512, f) != NULL) {
      char aliasname[256], refname[256];

      aliasname[0] = refname[0] = '\0';
      sscanf(line, "%s%s", aliasname, refname);
      if (aliasname[0] == '\0' || aliasname[0] == ';')
        continue; /* ignore empty lines or lines of comments */
      if (refname[0] == '\0') {
        printf("**Error** %s:%d: missing reference for alias '%s'\n",
          argv[0], linnum, aliasname);
        if (errors++ > 25) {
          printf("**Fatal** %s:%d: too many errors encountered !\n",
            argv[1], linnum);
          fatal = 2;
          break;
        }
      }
      OMFputALIAS(g, aliasname, refname);
    }
  }
  else {
    printf("**Fatal** "); perror(argv[2]);
  }
  fclose(f);
  if (!fatal && errors == 0) {
    OMFputCOMENT(g, COMF_PURGE, COMC_LINKPASS2, "\1", 1);
    OMFputMODEND(g, "");
  }
  fclose(g);
  if (fatal || errors != 0) {
    remove(argv[2]);
    if (fatal) exit(fatal);
    else       exit(1);
  }
  return 0;
}
